import { DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { TransformedAgencies } from "@/types/transformers";
import { ComponentWithClassName } from "@/types/utils";
import { zodResolver } from "@hookform/resolvers/zod";
import { useState } from "react";
import { useForm } from "react-hook-form";
import { z } from "zod";
import { addStatewideService } from "../lib/actions";
import { statewideServiceFormDataSchema } from "../zod/schema.form";
import { ErrorNoAgenciesContent } from "./error.no-agencies-content";
import { WorkflowFormStatewideServices } from "./workflow.form-statewide-services";
import { AppFormErrors } from "../type/errors";
import { revalidateGetStatewideServices } from "@/hooks/use-get-statewide-services";

export function WorkflowDialogAdd({ setClose, agencies, className }: ComponentWithClassName<{ setClose: () => void, agencies: TransformedAgencies }>) {
  const [errors, setErrors] = useState<AppFormErrors>(undefined)

  const form = useForm<z.infer<typeof statewideServiceFormDataSchema>>({
    resolver: zodResolver(statewideServiceFormDataSchema),
    defaultValues: {
      service_phone: undefined,
      service_url: undefined,
      service_message: "",
      service_agency: agencies[0]?.gtfsId,
      service_alias: "",
    }
  })

  if (!agencies || agencies.length === 0) {
    return (
      <ErrorNoAgenciesContent />
    )
  }

  return (
    <DialogContent className={className}>
      <DialogHeader>
        <DialogTitle>Add new service.</DialogTitle>
        <DialogDescription>
          Services will be visible for all users on the Trip Planner.
        </DialogDescription>
      </DialogHeader>

      <WorkflowFormStatewideServices
        form={form}
        agencies={agencies}
        errors={errors}
        overrides={{
          cancelText: "Discard new service",
          submitText: "Add service",
        }}
        handleSubmit={async (values) => {
          const res = await addStatewideService({
            url: values.service_url ?? null,
            phone: values.service_phone ?? null,
            alias: values.service_alias,
            message: values.service_message,
            agencyid: values.service_agency,
            status: "PUBLISHED",
          })

          await revalidateGetStatewideServices()

          if (!!res.errors && res.errors.length > 0) {
            setErrors(res.errors.map((e) => e.message))
            return;
          }

          setClose()
        }}
      />
    </DialogContent >
  )
}
